﻿using System;
using System.IO;
using System.Net.Http;
using System.Runtime.Serialization.Json;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Web;

namespace VRMRest
{

    /// <summary>
    /// 
    /// Instructions on adding to your plugin,
    /// 1) Add a an existing Item to your project, navigate to this cs file, down in the bottom right corner where it says "Add", change to "Add as Link"
    /// 2) Add a Reference to System.Net.Http to the plugin project this is needed by HttpClient
    /// 3) Add a Reference to System.Xml to the plugin project this is needed by the Json Deserializer.
    /// 
    /// </summary>
    public class Utility
    {
        public enum LogLevel { Debug = 935950000, Info = 935950001, Warn = 935950002, Error = 935950003, Fatal = 935950004, Timing = 935950005 };

        public const string OneWayPassTest = "TestMessages#OneWayPassTest";
        public const string TwoWayPassTest = "TestMessages#TwoWayPassTest";
        public const string CreateCRMLogEntryRequest = "CRMe#CreateCRMLogEntryRequest";

        const string _urlRestPath = "/Servicebus/Rest/{0}";
        const string _urlParams = "?messageId={0}&messageType=text%2Fjson&isQueued=false";
        
        const string SEND = "Send";
        const string SEND_RECEIVE = "SendReceive";

        /// <summary>
        /// Send a Request Message to VIMT that has no Response
        /// </summary>
        /// <param name="baseUri"></param>
        /// <param name="messageId"></param>
        /// <param name="requestObj"></param>
        /// <returns></returns>
        public static HttpResponseMessage Send(Uri baseUri, string messageId, object requestObj, LogSettings logSettings)
        {
            HttpResponseMessage responseMessage = null;

            Uri uri = FormatUri(baseUri, SEND, messageId);
            try
            {
                using (HttpClient client = new HttpClient())
                {

                    string message = string.Empty; ;

                    using (MemoryStream memStream = ObjectToJSonStream(requestObj))
                    {
                        using (StreamContent sc = new StreamContent(memStream))
                        {
                            responseMessage = client.PostAsync(uri, sc).Result;  
                            responseMessage.EnsureSuccessStatusCode();
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                if (logSettings != null)
                {
                    LogError(baseUri, logSettings, "Send", ex);
                }

                throw;
            }

            return responseMessage;
        }

        /// <summary>
        /// Send a Request Message Async to VIMT that has no Response 
        /// </summary>
        /// <param name="baseUri"></param>
        /// <param name="messageId"></param>
        /// <param name="obj"></param>
        /// <param name="logSettings"></param>
        /// <param name="callBack"></param>
        public static void SendAsync(Uri baseUri, string messageId, object obj, LogSettings logSettings, Action<HttpResponseMessage> callBack)
        {
            new Thread(() =>
                {
                    try
                    {
                        Thread.CurrentThread.IsBackground = false;
                        HttpResponseMessage response = Send(baseUri, messageId, obj, logSettings);
                        if (callBack != null)
                            callBack(response);
                    }
                    catch (Exception ex)
                    {
                        if (logSettings != null)
                        {
                            LogError(baseUri, logSettings, "SendAsync", ex);
                        }
                    }
                }).Start();

        }

        /// <summary>
        /// Send Request Message to VIMT and return a Response of type T
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="baseUri"></param>
        /// <param name="messageId"></param>
        /// <param name="obj"></param>
        /// <returns></returns>
        public static T SendReceive<T>(Uri baseUri, string messageId, object obj, LogSettings logSettings)
        {
            string message = string.Empty;
            Uri uri = FormatUri(baseUri, SEND_RECEIVE, messageId);

            try
            {
                using (HttpClient client = new HttpClient())
                {

                    using (MemoryStream memStream = ObjectToJSonStream(obj))
                    {
                        string responseValue;

                        using (StreamContent sc = new StreamContent(memStream))
                        {
                            //.Result will wait for these Async calls to finish.
                            HttpResponseMessage response = client.PostAsync(uri, sc).Result; 
                            response.EnsureSuccessStatusCode();
                            responseValue = response.Content.ReadAsStringAsync().Result; 
                        }

                        //Parse message from VIMT, if there are valid tags.
                        int start = responseValue.IndexOf("<Message>") + 9;
                        int end = responseValue.IndexOf("</Message>");

                        if (end != -1)
                        {

                            message = responseValue.Substring(start, end - start);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                if (logSettings != null)
                {
                    LogError(baseUri, logSettings, "SendReceive", ex);
                }
                throw;
            }
            T retObj = DeserializeResponse<T>(message);
            return retObj;
        }

        /// <summary>
        /// Format the URI with method and Message type.
        /// </summary>
        /// <param name="baseUri"></param>
        /// <param name="method"></param>
        /// <param name="messageId"></param>
        /// <returns></returns>
        public  static Uri FormatUri(Uri baseUri, string method, string messageId)
        {
            string urlRestPath = string.Format(_urlRestPath, method);
            string urlParams = string.Format(_urlParams, HttpUtility.UrlEncode(messageId));
            Uri relativeUri = new Uri(urlRestPath + urlParams, UriKind.Relative);
            Uri retUri = new Uri(baseUri, relativeUri);
            return retUri;

        }

        /// <summary>
        /// Deserialize Message to type T
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="message"></param>
        /// <returns></returns>
        private static T DeserializeResponse<T>(string message)
        {
            T retObj;
            byte[] b = Convert.FromBase64String(message);
            UTF8Encoding enc = new UTF8Encoding();
            string mess = enc.GetString(b);

            //REplace out the NewtonSoft specific dates with datacontract dates.
            string fixedDates = Regex.Replace(mess, @"new Date\(([-+0-9]*)\)", "\"\\/Date($1)\\/\"");

            using (MemoryStream ms = new MemoryStream())
            {
                ms.Write(enc.GetBytes(fixedDates), 0, fixedDates.Length);

                ms.Position = 0;

                DataContractJsonSerializer ser = new DataContractJsonSerializer(typeof(T));
                retObj = (T)ser.ReadObject(ms);
            }
            return retObj;
        }

        /// <summary>
        /// Convert object to MemoryStream in Json.
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        private static MemoryStream ObjectToJSonStream(object obj)
        {
            MemoryStream memStream = new MemoryStream();

            DataContractJsonSerializer ser = new DataContractJsonSerializer(obj.GetType());

            ser.WriteObject(memStream, obj);

            memStream.Position = 0;
            return memStream;
        }

        /// <summary>
        /// Log Error
        /// </summary>
        /// <param name="baseUri">REST URI to the VIMT Server</param>
        /// <param name="org">CRM Organization</param>
        /// <param name="configFieldName">Not Used: Future On/Off switch</param>
        /// <param name="userId">Calling UserId</param>
        /// <param name="method">Calling Method</param>
        /// <param name="message">Error Message</param>
        public static void LogError(Uri baseUri, string org, string configFieldName, Guid userId, string method, string message, string callingMethod)
        {
            try
            {
                CreateCRMLogEntryRequest logRequestStart = new CreateCRMLogEntryRequest()
                {
                    MessageId = Guid.NewGuid().ToString(),
                    OrganizationName = org,
                    UserId = userId,
                    crme_Name = string.Format("Exception: {0}:{1}", "Error in ", method),
                    crme_ErrorMessage = message,
                    crme_Debug = false,
                    crme_GranularTiming = false,
                    crme_TransactionTiming = false,
                    crme_Method = callingMethod + ":" + method,
                    crme_LogLevel = (int)LogLevel.Error,
                    crme_Sequence = 1,
                    NameofDebugSettingsField = configFieldName
                };

                CreateCRMLogEntryResponse logResponse = SendReceive<CreateCRMLogEntryResponse>(baseUri, Utility.CreateCRMLogEntryRequest, logRequestStart, null);
            }
            catch (Exception) { }

        }


        /// <summary>
        /// Log Error 
        /// </summary>
        /// <param name="baseUri">REST URI to the VIMT Server</param>
        /// <param name="org">CRM Organization</param>
        /// <param name="configFieldName">Not Used: Future On/Off switch</param>
        /// <param name="userId">Calling UserId</param>
        /// <param name="method">Calling Method</param>
        /// <param name="ex">Exception to log</param>
        public static void LogError(Uri baseUri, string org, string configFieldName, Guid userId, string method, Exception ex, string callingMethod)
        {
            string stackTrace = StackTraceToString(ex);
            LogError(baseUri, org, configFieldName, userId, method, stackTrace, callingMethod);
        }

        /// <summary>
        /// Log Error
        /// </summary>
        /// <param name="baseUri">REST URI to the VIMT Server</param>
        /// <param name="logSettings">LogSettings Object</param>
        /// <param name="method">Calling Method</param>
        /// <param name="ex">Exception to log</param>
        public static void LogError(Uri baseUri, LogSettings logSettings, string method, Exception ex)
        {
            LogError(baseUri, logSettings.Org, logSettings.ConfigFieldName, logSettings.UserId, method, ex, logSettings.callingMethod);
        }

        /// <summary>
        /// concatentate message and stack traces for exceptions and subsequent innerexceptions.
        /// </summary>
        /// <param name="ex"></param>
        /// <returns></returns>
        public static string StackTraceToString(Exception ex)
        {
            StringBuilder sb = new StringBuilder();
            sb = StackTraceToString(ex, sb);
            return sb.ToString();
        }

        /// <summary>
        /// Recursive call to concatentate message and stack traces for exceptions and subsequent innerexceptions.
        /// </summary>
        /// <param name="ex"></param>
        /// <param name="sb"></param>
        /// <returns></returns>
        internal static StringBuilder StackTraceToString(Exception ex, StringBuilder sb)
        {

            sb.AppendLine("***************************");
            sb.AppendLine(ex.Message);
            sb.AppendLine(ex.StackTrace);

            if (ex.InnerException != null)
            {
                sb = StackTraceToString(ex.InnerException, sb);
            }

            return sb;
        }


    }


    public class LogSettings
    {
        /// <summary>
        /// CRM Organization
        /// </summary>
        public string Org { get; set; }
        /// <summary>
        /// Future Use On/Off Switch
        /// </summary>
        public string ConfigFieldName { get; set; }
        /// <summary>
        /// Calling User
        /// </summary>
        public Guid UserId { get; set; }
        /// <summary>
        /// Calling Method
        /// </summary>
        public string callingMethod { get; set; }
    }

    #region Log Messages

    public class CreateCRMLogEntryRequest 
    {
        public string MessageId { get; set; }

        public string OrganizationName { get; set; }
        
        public Guid UserId { get; set; }
        
        public int crme_Sequence { get; set; }
        
        public string crme_Name { get; set; }
        
        public string NameofDebugSettingsField { get; set; }
        
        public string crme_ErrorMessage { get; set; }
        
        public string crme_Method { get; set; }
        
        public bool crme_GranularTiming { get; set; }
        
        public bool crme_TransactionTiming { get; set; }
        
        public bool crme_Debug { get; set; }
        
        public int crme_LogLevel { get; set; }
        
        public Guid crme_RelatedParentId { get; set; }
        
        public string crme_RelatedParentEntityName { get; set; }
        
        public string crme_RelatedParentFieldName { get; set; }
        
        public string crme_RelatedWebMethodName { get; set; }
        
        public string crme_TimeStart { get; set; }
        
        public string crme_TimeEnd { get; set; }
        
        public Decimal crme_Duration { get; set; }
    }

    public class CreateCRMLogEntryResponse 
    {
        public string MessageId { get; set; }
        public Guid crme_loggingId { get; set; }
    }



    #endregion



}
